(* ::Subsubsection:: *)
(*BuchroederCamera package*)

(* ::Text:: *)
(*This impements the Buchroeder Camera optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/BuchroederCamera"}]*)

BeginPackage["GeometricOptics`Packages`BuchroederCamera`", {"GeometricOptics`"}]

Options[BuchroederCamera] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
BuchroederCamera::usage="BuchroederCamera calculates parameters for a Buchroeder Camera optical system";

BuchroederCameraInputPanel::usage = "BuchroederCameraInputPanel gives the GUI panel to work with BuchroederCamera";

$BuchroederCameraInputVariables = {"f1", "spes", "diam", "Ni", "\[Theta]"};

Begin["`Private`"]

BuchroederCamera[{f1_, spes_, diam_, Ni_, theta_}, opts___]:= BuchroederCamera[f1, spes, diam, Ni, theta, opts];

BuchroederCamera[f1_, spes_, diam_, Ni_, theta_, OptionsPattern[]] := 
 Module[{DS1, R10, sol, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["BuchroederCamera", {$BuchroederCameraInputVariables, {f1, spes, diam, Ni, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 3, "PackageName" -> "BuchroederCamera", "Arguments" -> <|"f1" -> f1, "spes" -> spes, "diam" -> diam, "Ni" -> Ni, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];

			DS1 = -1/(32 f1^3);
			R10 = (2 2^(1/3) f1 (-1 + Ni)^(2/3) (1 + Ni)^(1/3))/Ni^(1/3);
			(* Elimination of spherical aberrations for thick lenses *)
			TotalAberrations[{R1, Infinity, -R1, R1, Infinity, -R1, -2 f1}, Join[spes, {2 f1}], {{1, Ni, 1, Ni, 1, Ni, 1, -1}}, {0, 0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			sol = FindRoot[GOaberration[[1]] == 0, {R1, R10}, AccuracyGoal -> 4];
			
			outputs = { {"Primary radius", -2 f1},
						{"Distance between primary and corrector = ", 2 f1},
						{"The radii of the triplet are", ""},
						{"r1 ", R1 /. sol},
						{"r2 ", "\[Infinity]"},
						{"r3 ", -R1 /. sol},
						{"r4 ", R1 /. sol},
						{"r5 ", "\[Infinity]"},
						{"r6 ", -R1 /. sol},
						{"Height of the object", GOimageHeight[[1, GOn]] /. sol},
						{"Focal length", GOfocalLength[[1]] /. sol}, 
						{"Total spherical aberration", GOaberration[[1]] /. sol},
						{"Total sagittal coma", GOcoma[[1]] /. sol},
						{"Total astigmatism", GOastigmatism[[1]] /. sol}}//N;

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"List of thickness of the lenses", "spes", spes},
							{"Diameter of primary mirror", "diam", diam},
							{"Refractive index of the elens", "Ni", Ni},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BuchroederCamera" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "BuchroederCamera", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BuchroederCamera" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "BuchroederCamera", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
											 {"spes", spes},
											 {"diam", diam},
											 {"Ni", Ni},
											 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[
					TextCell[{-2 f1, 2f1, R1 /. sol, \[Infinity], -R1 /. sol, R1 /. sol, \[Infinity], -R1 /. sol, GOimageHeight[[1, GOn]] /. sol, GOfocalLength[[1]] /. sol, 
							  GOaberration[[1]] /. sol, GOcoma[[1]] /. sol, GOastigmatism[[1]] /. sol}, "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["BuchroederCamera not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

BuchroederCameraInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "BuchroederCamera"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, spes, diam, Ni, theta} = ReplaceAll[$BuchroederCameraInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"List of thickness of the lenses", "spes", Tooltip[InputField[Dynamic[spes], Alignment -> Center], "Insert the list of thickness for each lens"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Refractive index for the lens", "Ni", Tooltip[InputField[Dynamic[Ni], Alignment -> Center], "Input the value of Ni"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, spes, diam, Ni, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, spes, diam, Ni, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]